<?php
/**
 * Database.php
 *
 * The Database class is meant to send and
 * get information from the database.
 *
 * By dees040 (https://github.com/dees040)
 * V0.1 on 01 february 2014
 * V1.0 on 08 february 2014
 * V2.0 on 07 may 2014
 */

include 'constants.php';
include 'error_handler.class.php';

class Database extends errorHandler
{
    private static $connection; // Database connection holder
    private static $debugging; // Debugging variable ON or OFF
    private static $errorType; // The error type SMALL, NORMAL or BIG

    /**
     * __construct - Class constructor.
     */
    public function __construct($debugging = OFF) {
        self::init();

        // Set debugging on or off
        if ($debugging == DEBUGGING) {
            self::$debugging = ON;
        } else {
            self::$debugging = OFF;
        }

        self::$errorType = NORMAL; // Set errorType to normal standard
    }

    /**
     * init - Initializer, create database connection.
     */
    private static function init() {
        try {
            self::$connection = new PDO('mysql:host='.DB_SERVER.';dbname='.DB_NAME, DB_USER, DB_PASS);
            self::$connection->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        } catch(PDOException $e) {
            echo "Error while connecting to ".DB_NAME.": ".$e->getMessage();
        }
    }

    /**
     * query - Function that create queries.
     * $query:     The query to execute
     * $items:     Array of parameters
     * $do:        Fetch database from query (FETCH, FETCH_ALL, FETCH_OBJECT, FETCH_ALL_OBJECT)
     * $errorType: The type of error to display on failure
     *
     * Returns the Query/PDO object.
     */
    public function query($query, $items = array(), $do = OFF, $errorType = NORMAL) {
        self::$errorType = $errorType;
        $createdQuery = self::queryChecker($query, $items);

        if (!$createdQuery) return false;

        if ($do != OFF) return self::fetcher($createdQuery, $do);

        return $createdQuery;
    }

    /**
     * queryChecker - Function that checks what to do with a given query.
     *
     * $query: The query
     * $items: Array of parameters
     */
    private static function queryChecker($query, $items) {
        $createdQuery = NULL;


        if (count(explode(" ", $query)) == 1) {
            $createdQuery = self::createQuery("SELECT * FROM ".$query, $items);
        } else {
            $createdQuery = self::createQuery($query, $items);
        }

        return $createdQuery;
    }

    /**
     * createQuery - Function that execute the query.
     *
     * $query: The query
     * $items: Array of parameters
     *
     * Returns PDO object.
     */
    private static function createQuery($query, $items) {
        try {
            $stmt = self::$connection->prepare($query);
            $stmt->execute($items);
        } catch(PDOException $e) {
            $stmt = NULL;
            self::errorHandling($e, self::$errorType);
        }

        return $stmt;
    }

    /**
     * fetcher - Function that fetch data from Query (PDO) Objects.
     *
     * $pdoStatement: Query/PDO Object
     * $fetchType:    Type how to fetch the data
     *
     * Return fetched data. (Object or Array), false on failure.
     */
    private static function fetcher($pdoStatement, $fetchType) {
        $fetch = NULL;

        if (!$pdoStatement instanceof PDOStatement) return false;

        switch($fetchType) {
            case FETCH:
                $fetch = $pdoStatement->fetch();
                break;
            case FETCH_ALL:
                $fetch = $pdoStatement->fetchAll();
                break;
            case FETCH_OBJECT:
                $fetch = $pdoStatement->fetchObject();
                break;
            case FETCH_OBJECT_ALL:
                $fetch = (object)$pdoStatement->fetchAll();
                break;
            default:
                $fetch = $pdoStatement->fetch();
        }

        return $fetch;
    }

    /**
     * errorHandling - Function that check if debugging is on.
     * If debugging is on it will display the given error.
     *
     * $e:         The PDOException error
     * $errorType: The errorType
     */
    private  static function errorHandling($e, $errorType = NORMAL) {
        if (self::$debugging == ON) {
            echo parent::createError($e, $errorType);
        }
    }

    /**
     * conn - Function that, if a PDO object is given,
     * sets the PDO connection and returns the connection.
     */
    public function conn($conn = NULL) {
        if ($conn instanceof PDO) {
            self::$connection = $conn;
        }

        return self::$connection;
    }
}

// Query class, create queries
class Query extends errorHandler
{
    private static $connection;
    private static $query;
    private static $items = array();
    private static $errorType;

    /**
     * __construct - Class constructor.
     *
     * $query:     Query to set
     * $items:     Array of parameters that belongs to the Query
     * $debugging: Debugging settings (ON or OFF)
     */
    public function __construct($query, $items = array(), $debugging = OFF) {
        self::init();

        if ($debugging == DEBUGGING) {
            self::$debugging = ON;
        } else {
            self::$debugging = OFF;
        }

        self::$errorType = NORMAL;

        self::initQuery($query, $items);
    }

    /**
     * init - Create database connection.
     */
    private static function init() {
        try {
            self::$connection = new PDO('mysql:host='.DB_SERVER.';dbname='.DB_NAME, DB_USER, DB_PASS);
            self::$connection->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        } catch(PDOException $e) {
            echo "Error while connecting to ".DB_NAME.": ".$e->getMessage();
        }
    }

    /**
     * initQuery - Function that checks what to do with the given Query.
     *
     * $query: The query
     * $items: Array of parameters that belongs to the Query
     */
    private static function initQuery($query, $items) {
        if (empty($items)) {
            self::$query = $query;
        } else {
            self::$query = $query;
            self::$items = $items;
            self::execute();
        }
    }

    /**
     * create - Function that create Queries.
     *
     * $query: The query
     * $items: Array of parameters that belongs to the Query
     */
    public function create($query, $items) {
        if (empty($items)) {
            self::$query = $query;
        } else {
            self::$query = $query;
            self::$items = $items;
        }
    }

    /**
     * bindParam - Function that bind parameters to the query.
     *
     * $key: The key of the parameter, can be an array of parameters as well
     * $val: The val of the parameter
     */
    public function bindParam($key, $value = "") {
        if (is_array($key)) {
            self::$items = $key;
            return true;
        } else if ($value) {
            self::$items[$key] = $value;
            return true;
        }

        return false;
    }

    /**
     * execute - Execute the Query.
     */
    public function execute() {
        try {
            $stmt = self::$connection->prepare(self::$query);
            $stmt->execute(self::$items);
        } catch(PDOException $e) {
            $stmt = NULL;
            self::errorHandling($e, self::$errorType);
        }

        return $stmt;
    }

    /**
     * errorHandling - Function that check if debugging is on.
     * If debugging is on it will display the given error.
     *
     * $e:         The PDOException error
     * $errorType: The errorType
     */
    private static function errorHandling($e, $errorType) {
        if (self::$debugging == ON) {
            echo parent::createError($e, $errorType);
        }
    }
}